const express = require('express');
const bodyParser = require('body-parser');
const sqlite3 = require('sqlite3').verbose();
const jwt = require('jsonwebtoken');
const bcrypt = require('bcrypt');
const cors = require('cors');
const path = require('path');

const app = express();
const SECRET = 'change_this_secret_in_production';

app.use(bodyParser.json());
app.use(cors());
app.use('/static', express.static(path.join(__dirname, '..', 'frontend')));

// Initialize SQLite DB
const db = new sqlite3.Database('./app.db');

db.serialize(() => {
  db.run(`CREATE TABLE IF NOT EXISTS users (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    name TEXT,
    email TEXT UNIQUE,
    password TEXT,
    role TEXT,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP
  )`);
  db.run(`CREATE TABLE IF NOT EXISTS records (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    user_id INTEGER,
    title TEXT,
    description TEXT,
    status TEXT DEFAULT 'open',
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY(user_id) REFERENCES users(id)
  )`);
});

// Create default admin if not exists
const DEFAULT_ADMIN_EMAIL = 'admin@example.com';
const DEFAULT_ADMIN_PASS = 'Admin@123'; // change after first run

db.get('SELECT * FROM users WHERE email = ?', [DEFAULT_ADMIN_EMAIL], (err,row)=>{
  if(err) console.error(err);
  if(!row){
    bcrypt.hash(DEFAULT_ADMIN_PASS, 10).then(hash=>{
      db.run('INSERT INTO users (name,email,password,role) VALUES (?,?,?,?)',
        ['Administrator', DEFAULT_ADMIN_EMAIL, hash, 'admin']);
      console.log('Default admin created ->', DEFAULT_ADMIN_EMAIL, DEFAULT_ADMIN_PASS);
    });
  }
});

// Auth middleware
function authenticate(req,res,next){
  const header = req.headers['authorization'];
  if(!header) return res.status(401).json({error:'No token'});
  const token = header.split(' ')[1];
  if(!token) return res.status(401).json({error:'Malformed token'});
  jwt.verify(token, SECRET, (err, decoded)=>{
    if(err) return res.status(401).json({error:'Invalid token'});
    req.user = decoded;
    next();
  });
}

function requireRole(role){
  return (req,res,next)=>{
    if(!req.user) return res.status(401).json({error:'Not authenticated'});
    if(req.user.role !== role) return res.status(403).json({error:'Forbidden: requires '+role});
    next();
  }
}

// Routes
app.post('/api/login', (req,res)=>{
  const {email,password} = req.body;
  if(!email || !password) return res.status(400).json({error:'Missing fields'});
  db.get('SELECT * FROM users WHERE email = ?', [email], (err,user)=>{
    if(err) return res.status(500).json({error:err.message});
    if(!user) return res.status(400).json({error:'Invalid credentials'});
    bcrypt.compare(password, user.password).then(match=>{
      if(!match) return res.status(400).json({error:'Invalid credentials'});
      const token = jwt.sign({id:user.id, name:user.name, email:user.email, role:user.role}, SECRET, {expiresIn:'8h'});
      res.json({token, user:{id:user.id, name:user.name, email:user.email, role:user.role}});
    });
  });
});

app.post('/api/register', authenticate, requireRole('admin'), (req,res)=>{
  const {name,email,password,role} = req.body;
  if(!name||!email||!password||!role) return res.status(400).json({error:'Missing fields'});
  bcrypt.hash(password, 10).then(hash=>{
    db.run('INSERT INTO users (name,email,password,role) VALUES (?,?,?,?)',[name,email,hash,role], function(err){
      if(err) return res.status(500).json({error:err.message});
      res.json({id:this.lastID, name, email, role});
    });
  });
});

app.get('/api/users', authenticate, (req,res)=>{
  // admins see all; operators see only themselves
  if(req.user.role === 'admin'){
    db.all('SELECT id,name,email,role,created_at FROM users', [], (err,rows)=>{
      if(err) return res.status(500).json({error:err.message});
      res.json(rows);
    });
  } else {
    db.get('SELECT id,name,email,role,created_at FROM users WHERE id = ?', [req.user.id], (err,row)=>{
      if(err) return res.status(500).json({error:err.message});
      res.json([row]);
    });
  }
});

// Records CRUD
app.post('/api/records', authenticate, (req,res)=>{
  const {title,description} = req.body;
  if(!title) return res.status(400).json({error:'Title required'});
  const user_id = req.user.id;
  db.run('INSERT INTO records (user_id,title,description) VALUES (?,?,?)', [user_id,title,description], function(err){
    if(err) return res.status(500).json({error:err.message});
    res.json({id:this.lastID, user_id, title, description});
  });
});

app.get('/api/records', authenticate, (req,res)=>{
  if(req.user.role === 'admin'){
    db.all('SELECT r.*, u.name as user_name FROM records r LEFT JOIN users u ON u.id=r.user_id ORDER BY r.created_at DESC', [], (err,rows)=>{
      if(err) return res.status(500).json({error:err.message});
      res.json(rows);
    });
  } else {
    db.all('SELECT r.*, u.name as user_name FROM records r LEFT JOIN users u ON u.id=r.user_id WHERE r.user_id = ? ORDER BY r.created_at DESC', [req.user.id], (err,rows)=>{
      if(err) return res.status(500).json({error:err.message});
      res.json(rows);
    });
  }
});

app.put('/api/records/:id', authenticate, (req,res)=>{
  const id = req.params.id;
  const {title,description,status} = req.body;
  // Check ownership or admin
  db.get('SELECT * FROM records WHERE id = ?', [id], (err,rec)=>{
    if(err) return res.status(500).json({error:err.message});
    if(!rec) return res.status(404).json({error:'Not found'});
    if(req.user.role !== 'admin' && rec.user_id !== req.user.id) return res.status(403).json({error:'Not allowed'});
    db.run('UPDATE records SET title = COALESCE(?, title), description = COALESCE(?, description), status = COALESCE(?, status) WHERE id = ?', [title,description,status,id], function(err){
      if(err) return res.status(500).json({error:err.message});
      res.json({updated:true});
    });
  });
});

app.delete('/api/records/:id', authenticate, (req,res)=>{
  const id = req.params.id;
  db.get('SELECT * FROM records WHERE id = ?', [id], (err,rec)=>{
    if(err) return res.status(500).json({error:err.message});
    if(!rec) return res.status(404).json({error:'Not found'});
    if(req.user.role !== 'admin' && rec.user_id !== req.user.id) return res.status(403).json({error:'Not allowed'});
    db.run('DELETE FROM records WHERE id = ?', [id], function(err){
      if(err) return res.status(500).json({error:err.message});
      res.json({deleted:true});
    });
  });
});

// Serve frontend index
app.get('/', (req,res)=>{
  res.sendFile(path.join(__dirname, '..', 'frontend', 'index.html'));
});

const PORT = process.env.PORT || 3000;
app.listen(PORT, ()=> console.log('Server running on port', PORT));
